/*-------------------------------------------------------------------------------*
 * File Name:	ozlib.h	   														 *
 * Creation:	09/08/2004														 *
 * Purpose: interface of the 'zlib' general purpose compression library,         *
 *          which is compatible with the gzip format							 *
 * Copyright (c) OriginLab Corp.2004											 *
 * All Rights Reserved															 *
 *																				 *
 * Acknowledgment:Thanks to Jean-Loup Gailly and Mark Adler for the zlib library *											*
 * 																				 *
 * Modification Log:															 *
 *-------------------------------------------------------------------------------*/

#ifndef _OZLIB_H
#define _OZLIB_H

#define Z_OK            0
#define Z_DATA_ERROR   (-3)
#define Z_MEM_ERROR    (-4)
#define Z_BUF_ERROR    (-5)

#pragma dll(zlibwapi)	

#define ZLIB_API WINAPI

/**$
   Compresses the source buffer into the destination buffer.
   This function can be used to compress a whole file at once if the input file is mmap'ed.
   
   Example:
   		This example reads the first 1024 bytes from file source.dat, then compresses the bytes
   		and write the compressed data into compress.dat.
   		source.dat and compress.dat must exist, if the length of source.dat is less than 1K,
   		the whole file source.dat will be compressed into compress.dat.
   		
			#include <stdio.h>
			void	test_compress()
			{
				FILE *streamIn, *streamOut;
				if( ( streamIn = fopen( "source.dat", "r" ) ) == NULL )
				{
					printf( "Problem opening input file\n" );
					return;
				}
				if( ( streamOut = fopen( "compress.dat", "w" ) ) == NULL )
				{
					printf( "Problem opening output file\n" );
					return;
				}
				byte source[1024], dest[1024];
				unsigned long nSourceLen = fread( source, sizeof( byte ), 1024, streamIn );
				unsigned long nDestLen = 1024;
				int	iRet =  compress( dest, &nDestLen, source, nSourceLen );
				if( iRet )
				{
					printf( "Compression error\n" );
					return;
				}
				fwrite( dest, sizeof( byte ), nDestLen, streamOut );
				fclose( streamIn );
				fclose( streamOut );
			}
   	
   Parameters:
		dest:	   the destination buffer
   		destLen:   Upon entry, destLen is the total size of the destination buffer,
   				   which must be at least the value returned by compressBound(sourceLen).
   				   Upon exit, destLen is the actual size of the compressed buffer.
   		source:	   the source buffer
   		sourceLen: the byte length of the source buffer. 
   Return:
		Z_OK:	   	  success
		Z_MEM_ERROR:  there was not enough memory
		Z_BUF_ERROR:  there was not enough room in the output buffer
*/   
int ZLIB_API compress(byte *dest, unsigned long *destLen, const byte *source, unsigned long sourceLen);
  
/**$
   Decompresses the source buffer into the destination buffer.
   This function can be used to decompress a whole file at once if the input file is mmap'ed.
   
   Example:
   		This example reads the first 1024 bytes from file source.dat, then compresses the bytes
   		and write the compressed data into compress.dat.
   		The compressed data is uncompressed and written into uncompress.dat, which should be the same as the
   		first 1K bytes of source.dat.
   		source.dat, compress.dat, uncompress.dat must exist, if the length of source.dat is less than 1K,
   		the whole file source.dat will be compressed into compress.dat, and uncompress.dat should be the same as source.dat.
   		
   			#include <stdio.h>
			void	test_uncompress()
			{
				FILE *streamIn, *streamComp, *streamUncomp;
				if( ( streamIn = fopen( "source.dat", "r" ) ) == NULL )
				{
					printf( "Problem opening input file\n" );
					return;
				}
				if( ( streamComp = fopen( "compress.dat", "w" ) ) == NULL )
				{
					printf( "Problem opening compress file\n" );
					return;
				}
				if( ( streamUncomp = fopen( "uncompress.dat", "w" ) ) == NULL )
				{
					printf( "Problem opening uncompress file\n" );
					return;
				}
				byte source[1024], comp[1024], uncomp[2048];
				unsigned long nSourceLen = fread( source, sizeof( byte ), 1024, streamIn );
				unsigned long nCompLen = 1024;
				unsigned long nUncompLen = 2048;
				int	iRet;
				
				iRet =  compress( comp, &nCompLen, source, nSourceLen );
				if( iRet )
				{
					printf( "Compression error\n" );
					return;
				}
				fwrite( comp, sizeof( byte ), nCompLen, streamComp );
							
				iRet =  uncompress( uncomp, &nUncompLen, comp, nCompLen);
				if( iRet )
				{
					printf( "Uncompression error\n" );
					return;
				}
				fwrite( uncomp, sizeof( byte ), nUncompLen, streamUncomp );
				
				fclose( streamIn );
				fclose( streamComp );
				fclose( streamUncomp );
			}
   	
   Parameters:
		dest:	   the destination buffer
   		destLen:   Upon entry, destLen is the total size of the destination buffer, 
   				   which must be large enough to hold the entire uncompressed data.
   				   (The size of the uncompressed data must have been saved previously by
   				   the compressor and transmitted to the decompressor by some mechanism
   				   outside the scope of this compression library.)
   				   Upon exit, destLen is the actual size of the compressed buffer.
   		source:	   the source buffer
   		sourceLen: the byte length of the source buffer. 
   Return:
		Z_OK:	   	  success
		Z_MEM_ERROR:  there was not enough memory
		Z_BUF_ERROR:  there was not enough room in the output buffer
		Z_DATA_ERROR: the input data was corrupted or incomplete
*/                               
int ZLIB_API uncompress(byte *dest, unsigned long *destLen, const byte *source, unsigned long sourceLen);

#endif	//_OZLIB_H